// SPDX-FileCopyrightText: 2023, 2024, 2025 g10 code GmbH
// SPDX-Contributor: Carl Schwan <carl.schwan@gnupg.com>
// SPDX-Contributor: Sune Stolborg Vuorela <sune@vuorela.dk>
// SPDX-License-Identifier: GPL-2.0-or-later

#include "truststore.h"
#include "controller.h"

#include <QString>

#ifdef Q_OS_WIN
#include <QCoreApplication>
#include <QDebug>
#include <QDir>
#include <QIODevice>
#include <QTemporaryFile>
#include <windows.h>
// include after windows.h
#include <shellapi.h>
#endif
#ifdef Q_OS_LINUX
#include <QFileInfo>
#include <QProcess>
#include <QStandardPaths>

#include <KAuth/ExecuteJob>
#endif

using namespace Qt::StringLiterals;

TrustStore::TrustStore() = default;
TrustStore::~TrustStore() = default;

QString TrustStore::systemTrustFilename(const Controller &controller) const
{
    return controller.caUniqueName().replace(u" "_s, u"_"_s);
}

bool TrustStore::uninstall(const Controller &controller)
{
    Q_UNUSED(controller);
    return true;
}

#ifdef Q_OS_LINUX
bool TrustStore::install(const Controller &controller)
{
    QVariantMap args;
    args["caUniqueName"_L1] = systemTrustFilename(controller);
    args["caCert"_L1] = controller.caCert();
    KAuth::Action installAction(u"com.gnupg.gpgolweb.truststore.install"_s);
    installAction.setHelperId(u"com.gnupg.gpgolweb.truststore"_s);
    installAction.setArguments(args);
    KAuth::ExecuteJob *job = installAction.execute();
    if (!job->exec()) {
        qDebug() << "KAuth returned an error code:" << job->error() << job->errorText();
    } else {
        QString contents = job->data()[u"contents"_s].toString();
        qDebug() << "KAuth succeeded. Contents: " << contents;
    }
    return true;
}
#endif

#ifdef Q_OS_WIN
bool TrustStore::install(const Controller &controller)
{
    QTemporaryFile file;
    file.open();
    file.setAutoRemove(false);
    file.write(controller.caCert());
    file.close();

    const QString args = u"-NoProfile -ExecutionPolicy Bypass -File \"%1\" CurrentUser \"%2\""_s.arg(
        QDir::toNativeSeparators(QCoreApplication::applicationDirPath() + u"/install.ps1"_s),
        QDir::toNativeSeparators(file.fileName()));

    HINSTANCE result = ShellExecuteW(nullptr, L"runas", L"powershell.exe", args.toStdWString().c_str(), nullptr, SW_SHOWNORMAL);

    if ((intptr_t)result <= 32) {
        DWORD errCode = GetLastError();

        // Format the error message
        LPWSTR errorMsg = nullptr;
        FormatMessageW(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS,
                       nullptr,
                       errCode,
                       0,
                       (LPWSTR)&errorMsg,
                       0,
                       nullptr);

        qDebug() << "ShellExecuteW failed with code:" << (intptr_t)result;
        qDebug() << "GetLastError():" << errCode;
        if (errorMsg) {
            qDebug() << "Error message:" << QString::fromWCharArray(errorMsg);
            LocalFree(errorMsg);
        }
    } else {
        qDebug() << "Root CA installation succeeded.";
    }

    return true;
}
#endif
