// SPDX-FileCopyrightText: 2024 g10 code Gmbh
// SPDX-Contributor: Carl Schwan <carl.schwan@gnupg.com>
// SPDX-License-Identifier: GPL-2.0-or-later

#include "firsttimedialog.h"
#include "config.h"
#include "gpgolweb_version.h"
#include "rootcagenerator/controller.h"
#include "ui_firsttimedialog.h"
#include "websocketclient.h"

#include <QCheckBox>
#include <QClipboard>
#include <QCloseEvent>
#include <QDesktopServices>
#include <QFile>
#include <QSaveFile>
#include <QStandardPaths>
#include <QStatusBar>
#include <QTemporaryDir>
#include <QToolBar>

#include <Libkleo/Compliance>

#include <KColorScheme>
#include <KIO/OpenFileManagerWindowJob>
#include <KTitleWidget>

using namespace Qt::StringLiterals;

FirstTimeDialog::FirstTimeDialog(QWidget *parent)
    : QMainWindow(parent)
    , ui(new Ui::FirstTimeDialog)
    , m_systemTrayIcon(QIcon::fromTheme(u"com.gnupg.gpgolweb"_s))
{
    ui->setupUi(this);
    ui->welcomePage->setProperty("title", i18nc("@title", "Welcome to GpgOL/Web"));
    ui->configurationPage->setProperty("title", i18nc("@title", "Welcome to GpgOL/Web"));
    ui->configPage->setProperty("title", i18nc("@title", "Install Outlook Add-In"));
    ui->installationPage->setProperty("title", i18nc("@title", "Setting Up the Local Server"));

    ui->reencryptOption->setChecked(Config::self()->reencrypt());
    connect(ui->reencryptOption, &QCheckBox::stateChanged, this, [](int state) {
        Config::self()->setReencrypt(state == Qt::Checked);
        Config::self()->save();
    });

    m_systemTrayIcon.setMainWindow(this);
    m_systemTrayIcon.show();

    m_backAction = new QAction(QIcon::fromTheme(u"draw-arrow-back-symbolic"_s), i18nc("@action:intoolbar", "Go back"));
    connect(m_backAction, &QAction::triggered, this, [this]() {
        if (Controller::certificateAlreadyGenerated() || !Config::self()->isLocalServer()) {
            ui->stack->setCurrentIndex(ui->stack->currentIndex() > 1 ? ConfigurationPage : WelcomePage);
        } else {
            ui->stack->setCurrentIndex(ConfigurationPage);
        }
    });
    auto toolbar = new QToolBar(this);
    toolbar->setMovable(false);
    toolbar->addAction(m_backAction);
    auto titleWidget = new KTitleWidget(this);
    toolbar->addWidget(titleWidget);
    addToolBar(Qt::TopToolBarArea, toolbar);

    m_backAction->setVisible(false);
    titleWidget->setText(ui->stack->currentWidget()->property("title").toString());
    connect(ui->stack, &QStackedWidget::currentChanged, this, [titleWidget, this]() {
        m_backAction->setVisible(ui->stack->currentIndex() > 0);
        titleWidget->setText(ui->stack->currentWidget()->property("title").toString());
    });

    // center vertically
    ui->mainLayout->insertStretch(0);
    ui->mainLayout->addStretch();

    ui->mainLayoutManifest->insertStretch(0);
    ui->mainLayoutManifest->addStretch();

    QPixmap logo = QIcon::fromTheme(u"com.gnupg.gpgolweb"_s).pixmap(64, 64);
    ui->logo->setPixmap(logo);
    ui->titleWelcome->setText(i18nc("@info", "GpgOL/Web %1", QString::fromLocal8Bit(GPGOLWEB_VERSION_STRING)));

    auto statusBar = new QStatusBar(this);

    auto showOnStartup = new QCheckBox(i18nc("@option:check", "Show on startup"));
    showOnStartup->setChecked(Config::self()->showLauncher());
    connect(showOnStartup, &QCheckBox::toggled, this, [](bool checked) {
        Config::self()->setShowLauncher(checked);
        Config::self()->save();
    });
    statusBar->addPermanentWidget(showOnStartup, 1);

    m_status = new QLabel;
    statusBar->addPermanentWidget(m_status);

    auto version = new QLabel(i18nc("@info", "Version: %1", QString::fromLocal8Bit(GPGOLWEB_VERSION_STRING)));
    statusBar->addPermanentWidget(version);

    if (Kleo::DeVSCompliance::isActive()) {
        auto statusLbl = std::make_unique<QLabel>(Kleo::DeVSCompliance::name());
        {
            auto statusPalette = qApp->palette();
            KColorScheme::adjustForeground(statusPalette,
                                           Kleo::DeVSCompliance::isCompliant() ? KColorScheme::NormalText : KColorScheme::NegativeText,
                                           statusLbl->foregroundRole(),
                                           KColorScheme::View);
            statusLbl->setAutoFillBackground(true);
            KColorScheme::adjustBackground(statusPalette,
                                           Kleo::DeVSCompliance::isCompliant() ? KColorScheme::PositiveBackground : KColorScheme::NegativeBackground,
                                           QPalette::Window,
                                           KColorScheme::View);
            statusLbl->setPalette(statusPalette);
        }
        statusBar->addPermanentWidget(statusLbl.release());
    }

    setStatusBar(statusBar);

    connect(ui->continueButton, &QPushButton::clicked, this, &FirstTimeDialog::slotSetup);
    connect(ui->configureButton, &QPushButton::clicked, this, [this]() {
        ui->stack->setCurrentIndex(ConfigurationPage);
    });

    ui->manifestPath->setText(QLatin1StringView(DATAROUTDIR) + u"/gpgol/manifest.xml"_s);

    connect(ui->openOutlookButton, &QPushButton::clicked, this, []() {
        QDesktopServices::openUrl(QUrl(u"https://outlook.office.com/mail/jsmvvmdeeplink/?path=/options/manageapps&bO=4"_s));
    });
    connect(ui->testPageButton, &QPushButton::clicked, this, [this]() {
        QDesktopServices::openUrl(QUrl(u"https://"_s + serverDomain() + u"/test"_s));
    });

    connect(ui->manifestPathCopy, &QPushButton::clicked, this, [this]() {
        QGuiApplication::clipboard()->setText(ui->manifestPath->text());
    });

    connect(ui->manifestPathOpenFolder, &QPushButton::clicked, this, [this]() {
        auto job = new KIO::OpenFileManagerWindowJob();
        job->setHighlightUrls({QUrl::fromUserInput(ui->manifestPath->text())});
        if (!qEnvironmentVariableIsEmpty("XDG_ACTIVATION_TOKEN")) {
            job->setStartupId(qgetenv("XDG_ACTIVATION_TOKEN"));
        }
        job->start();
    });

    ui->remoteLabel->setEnabled(!Config::self()->isLocalServer());
    ui->remoteServer->setEnabled(!Config::self()->isLocalServer());
    ui->remoteServer->setText(Config::self()->remoteAddress().toString());
    ui->remoteOption->setChecked(!Config::self()->isLocalServer());

    connect(ui->remoteOption, &QRadioButton::toggled, this, [this](bool checked) {
        Config::self()->setIsLocalServer(!checked);
        Config::self()->save();

        ui->remoteLabel->setEnabled(!Config::self()->isLocalServer());
        ui->remoteServer->setEnabled(!Config::self()->isLocalServer());
    });

    connect(ui->remoteServer, &QLineEdit::textChanged, this, [this]() {
        Config::self()->setRemoteAddress(QUrl::fromUserInput(ui->remoteServer->text()));
        Config::self()->save();
    });

    if (Controller::certificateAlreadyGenerated() || !Config::self()->isLocalServer()) {
        ui->stack->setCurrentIndex(WelcomePage);
        if (Controller::certificateAlreadyGenerated() && Config::self()->isLocalServer()) {
            startLocalServer();
        }
        startWebsocketClient();
    } else {
        ui->stack->setCurrentIndex(ConfigurationPage);
    }

    connect(&m_serverProcess, &QProcess::readyReadStandardError, this, [this]() {
        qWarning().noquote() << m_serverProcess.readAllStandardError();
    });

    connect(&m_serverProcess, &QProcess::readyReadStandardOutput, this, [this]() {
        qWarning().noquote() << m_serverProcess.readAllStandardOutput();
    });
}

FirstTimeDialog::~FirstTimeDialog() = default;

void FirstTimeDialog::slotStateChanged(const QString &stateDisplay)
{
    m_status->setText(stateDisplay);
}

void FirstTimeDialog::closeEvent(QCloseEvent *e)
{
    e->ignore();
    hide();
}

void FirstTimeDialog::slotSetup()
{
    delete m_controller;
    m_controller = new Controller(this);
    if (ui->localOption->isChecked()) {
        if (!Controller::certificateAlreadyGenerated()) {
            ui->stack->setCurrentIndex(InstallationPage);

            connect(m_controller, &Controller::result, this, [this](KJob *) {
                if (m_controller->error()) {
                    ui->installationPage->appendPlainText(m_controller->errorText());
                    return;
                }

                startLocalServer();
                startWebsocketClient();
                generateManifest();
            });
            connect(m_controller, &Controller::debutOutput, this, &FirstTimeDialog::slotTlsDebutOutput);
            m_controller->start();
        } else {
            startLocalServer();
            startWebsocketClient();
            generateManifest();
        }
    } else {
        generateManifest();
    }
}

void FirstTimeDialog::startLocalServer()
{
    if (m_serverProcess.state() != QProcess::NotRunning) {
        return;
    }

    m_serverProcess.start(u"gpgol-server"_s);
}

void FirstTimeDialog::startWebsocketClient()
{
    const auto clientId = QUuid::createUuid().toString(QUuid::WithoutBraces);
    auto &websocketClient = WebsocketClient::self(QUrl(u"wss://"_s + serverDomain() + u"/websocket"_s), clientId);

    connect(&websocketClient, &WebsocketClient::stateChanged, this, &FirstTimeDialog::slotStateChanged);
    connect(&websocketClient, &WebsocketClient::stateChanged, &m_systemTrayIcon, &SystemTrayIcon::slotStateChanged);

    m_systemTrayIcon.slotStateChanged(websocketClient.stateDisplay());
    slotStateChanged(websocketClient.stateDisplay());
}

void FirstTimeDialog::slotTlsDebutOutput(const QString &output)
{
    ui->installationPage->appendPlainText(output);
}

void FirstTimeDialog::generateManifest()
{
    QFile file(u":/gpgol-client/manifest.xml.in"_s);
    if (!file.open(QIODeviceBase::ReadOnly)) {
        Q_ASSERT(false);
        return;
    }

    ui->stack->setCurrentIndex(ManifestPage);

    QByteArray manifest = file.readAll();
    manifest.replace("%HOST%", serverDomain().toUtf8());
    manifest.replace("%VERSION%", GPGOLWEB_VERSION_STRING);

    const auto saveFilePath = QStandardPaths::writableLocation(QStandardPaths::AppLocalDataLocation) + u"/gpgol-web-manifest.xml"_s;

    QSaveFile saveFile(saveFilePath);
    if (!saveFile.open(QIODeviceBase::WriteOnly)) {
        Q_ASSERT(false);
        return;
    }
    saveFile.write(manifest);
    saveFile.commit();

    ui->manifestPath->setText(QDir::toNativeSeparators(saveFilePath));
}

QString FirstTimeDialog::serverDomain() const
{
    return ui->localOption->isChecked() ? u"localhost:5656"_s : ui->remoteServer->text();
}
